/**
 * 视图编辑器类
 */
class ViewDesign {
	"use strict"

	/**
	 * 构造器
	 * @param {String} elem				编辑器挂载节点
	 * @param {Object} constructInfo	编辑器构建数据
	 */
	constructor(elem = '', constructInfo = {}) {
		// 挂载节点
		this.elem = elem;
		// 视图编辑器jq对象
		this.designerElem = null;
		// 视图编辑设置
		this.setting = {
			// canvasWidth: '100%',	// 编辑器初始宽度
			// canvasHeight: '100%',	// 编辑器初始高度
			canvasWidth: 1600,		// 编辑器初始宽度
			canvasHeight: 1600,		// 编辑器初始高度
			paneWidth: 200,			// 窗体初始高度
			paneHeight: 260,		// 窗体初始宽度
			resizable: {			// 缩放
				tableList: {		// 左侧表和视图列表
					min: 100,		// 横向最小缩放
					max: 500		// 横向最大缩放
				},
				configuration: {	// 底部配置栏
					min: 100,		// 纵向最小缩放
					max: 400		// 纵向最大缩放
				},
			}
		};
		// 编辑器构建数据
		this.constructInfo = constructInfo;
		// 表和视图数据
		this.tableviewInfo = {};
		// 窗体字典
		this.paneDict = {};
		// 连接线字典
		this.lineDict = {};
		// 鼠标在画板中位置,-1表示鼠标离开画板
		this.mouseInCanvasPosition = { x: -1, y: -1 };
		// 底部配置栏数据
		this.configuration = { distinct: false, select: [], where: [], group: [], having: [], limit: {} };
	}

	/**
	 * 初始化视图编辑器
	 * @param {Function} callback	完成后回调函数
	 */
	init(callback) {
		this.checkConstructInfo(() => {
			this.configuration = { ...this.constructInfo.configuration };
			this.initDesigner(() => { this.bindDesigner(); callback && callback(); });
		});
	}

	/**
	 * 校验视图构建数据
	 * @param {Function} callback	完成后回调函数
	 */
	checkConstructInfo(callback) {
		// 初始化时没有节点时抛异常
		if (!this.elem) throw new Error('提供视图编辑器的dom！');
		this.getTableListInfo(() => {
			// 设置默认值
			const { pane = {}, line = {}, configuration = {} } = this.constructInfo;
			this.constructInfo = { pane, line, configuration };
			const { distinct = false, select = [], where = [], group = [], having = [], limit = {} } = this.constructInfo.configuration;
			this.constructInfo.configuration = { distinct, select, where, group, having, limit };
			// 有误的窗体编号
			let errorPaneIdList = [];
			// 判断窗体编号为数字
			for (const paneId in this.constructInfo.pane) {
				const {
					type,
					name,
					subName = '',
					top = 0,
					left = 0,
					width = this.setting.paneWidth,
					height = this.setting.paneHeight,
					checkedFields = []
				} = this.constructInfo.pane[paneId];
				if (!/^\d+$/.test(paneId) || !type || !name || !(this.tableviewInfo.tables.includes(name) || this.tableviewInfo.views.includes(name))) {
					errorPaneIdList.push(paneId);
					delete this.constructInfo.pane[paneId];
					continue;
				}
				// 设置默认值
				this.constructInfo.pane[paneId] = { type, name, subName, top, left, width, height, checkedFields };
			}
			// 判断连接线编号为数字，判断连接线是否连接有误的表和试图
			for (const lineId in this.constructInfo.line) {
				if (!/^\d+$/.test(lineId) || !Array.isArray(this.constructInfo.line[lineId]) || this.constructInfo.line[lineId].length != 2 || (() => {
					for (let i = 0; i < this.constructInfo.line[lineId].length; i++) {
						const { paneId, field } = this.constructInfo.line[lineId][i];
						if (!paneId || !field || errorPaneIdList.includes(paneId.toString())) return true;
						this.constructInfo.line[lineId][i] = { paneId, field };
					}
					return false;
				})()) {
					delete this.constructInfo.line[lineId];
					continue;
				}
			}
			// 判断底部配置栏是否查询有误的表和试图
			let tempSelect = [];
			for (let i = 0; i < this.constructInfo.configuration.select.length; i++) {
				const item = this.constructInfo.configuration.select[i];
				const { paneId, field, subField = '' } = item;
				if (!paneId || !field || errorPaneIdList.includes(paneId.toString())) {
					continue;
				}
				tempSelect.push({ paneId, field, subField });
			}
			this.constructInfo.configuration.select = tempSelect;
			callback && callback();
		});
	}

	/**
	 * 初始化编辑器
	 * 编辑器包括左侧表和视图列表、中间画板、底部配置栏、窗体、连接线
	 * @param {Function} callback	完成后回调函数
	 */
	initDesigner(callback) {
		const self = this;
		const designerElem = this.designerElem = $('<div class="view-design-main"></div>');
		$(this.elem).append(designerElem);

		// 初始化左侧表和视图列表
		initTableview();
		// 初始化中间画板
		initCanvas();
		// 初始化窗体组
		initPaneGroup(() => {
			// 初始化连接线组
			initLineGroup();
			// 初始化底部配置栏
			initConfiguration();
			// 完成回调
			callback && callback();
		});

		this.designerElem = $(this.elem).find('.view-design-main');

		// 初始化左侧表和视图列表
		function initTableview() {
			const tableListView = $('<div class="ui-widget-content tableview-view"></div>');
			const tableListItemView = $('<div class="tableview-item" data-type="table"></div>');

			designerElem.append(tableListView);

			self.tableviewInfo.tables.forEach(item => {
				tableListView.append(tableListItemView.clone().html(item).attr('data-type', 'table'));
			});
			self.tableviewInfo.views.forEach(item => {
				tableListView.append(tableListItemView.clone().html(item).attr('data-type', 'view'));
			});
		}

		// 初始化中间画板
		function initCanvas() {
			const canvasView = $(`
				<div class="canvas-view">
					<div class="canvas-main"></div>
				</div>
			`);
			if (self.setting?.canvasWidth) canvasView.find('.canvas-main').css('width', typeof self.setting.canvasWidth == 'number' ? self.setting.canvasWidth + 'px' : self.setting.canvasWidth);
			if (self.setting?.canvasHeight) canvasView.find('.canvas-main').css('height', typeof self.setting.canvasHeight == 'number' ? self.setting.canvasHeight + 'px' : self.setting.canvasHeight);
			designerElem.append(canvasView);
		}

		// 初始化窗体组
		function initPaneGroup(callback) {
			let paneIdList = Object.keys(self.constructInfo.pane);
			let index = 0;

			initPane();

			function initPane() {
				if (paneIdList.length == 0 || index >= paneIdList.length) return callback();
				const paneId = paneIdList[index];
				const paneConstructInfo = self.constructInfo.pane[paneId];
				self.createPane(parseInt(paneId), paneConstructInfo, () => {
					index++;
					initPane();
				});
			}
		}

		// 初始化连接线组
		function initLineGroup() {
			for (const lineId in self.constructInfo.line) {
				const lineConstructInfo = self.constructInfo.line[lineId];
				self.createLine(parseInt(lineId), lineConstructInfo);
			}
		}

		// 初始化底部配置栏
		function initConfiguration() {
			const configurationView = $(`
				<div class="ui-widget-content configuration-view">
					<div class="navbar">
						<div class="navbar-item navbar-item-this" data-type="SELECT">SELECT</div>
						<div class="navbar-item" data-type="FROM">FROM</div>
						<div class="navbar-item" data-type="WHERE">WHERE</div>
						<div class="navbar-item" data-type="GROUP">GROUP BY</div>
						<div class="navbar-item" data-type="HAVING">HAVING</div>
						<div class="navbar-item" data-type="LIMIT">LIMIT</div>
					</div>
					<div class="main">
						<div class="main-item main-item-this" data-type="SELECT">
							<div class="content"></div>
							<div class="tool">
								<span class="btn" data-type="create" style="font-size:18px;">+</span>
								<span class="btn" data-type="delete" style="font-size:18px;">-</span>
								<span class="btn" data-type="moveUp">↑</span>
								<span class="btn" data-type="moveDown">↓</span>
								<span class="btn" data-type="distinct">
									<input type="checkbox" id="DISTINCT">
									<label for="DISTINCT">DISTINCT</label>
								</span>
							</div>
						</div>
						<div class="main-item" data-type="FROM">
							<div class="content">FROM</div>
						</div>
						<div class="main-item" data-type="WHERE">
							<div class="content">WHERE</div>
						</div>
						<div class="main-item" data-type="GROUP">
							<div class="content">GROUP BY</div>
						</div>
						<div class="main-item" data-type="HAVING">
							<div class="content">HAVING</div>
						</div>
						<div class="main-item" data-type="LIMIT">
							<div class="content">LIMIT</div>
						</div>
					</div>
				</div>
			`);

			designerElem.append(configurationView);

			configurationView.find('.main-item[data-type="SELECT"] .tool #DISTINCT').prop('checked', self.configuration.distinct);
			self.configuration.select.forEach((item) => {
				// const pane = self.paneDict[item.paneId];
				// configurationView.find('.main-item[data-type="SELECT"] .content').append(`
				// 	<div class="content-item" data-pane="${item.paneId}" data-field="${item.field}">
				// 		<span class="table-name">${pane.subName || pane.name}.${item.field}</span>
				// 		<span class="sub-field-name">${item.subField ? `(${item.subField})` : ''}</span>
				// 		<div class="delete-item">x</div>
				// 	</div>
				// `);
				self.createConfigurationSelect(item.paneId, item.field, item.subField, false);
			});
		}
	}

	/**
	 * 监听编辑器事件
	 */
	bindDesigner() {
		const tableListElem = this.designerElem.find('.tableview-view');
		const canvasElem = this.designerElem.find('.canvas-view');
		const configurationElem = this.designerElem.find('.configuration-view');

		// 左侧表和视图列表缩放事件
		tableListElem.resizable({
			handles: 'e',
			minWidth: this.setting.resizable.tableList.min,
			maxWidth: this.setting.resizable.tableList.max,
			resize: (event, ui) => {
				this.designerElem.find(".canvas-view").css('left', (ui.size.width + 1) + 'px');
				this.designerElem.find(".configuration-view").css('left', (ui.size.width + 1) + 'px');
			}
		});

		// 绑定拖拽事件
		tableListElem.find('.tableview-item').each((index, elem) => {
			// 拖拽列表中明细项
			$(elem).draggable({
				cursor: "move",				// 拖拽方式
				revert: "invalid",			// 还原方式
				helper: (event) => {		// 视觉反馈
					return $(`
						<div class="pane-box" style="width:200px;height:260px;">
							<div class="header">
								<span class="title">${$(elem).html()}</span>
							</div>
							<div class="fields"></div>
						</div>
					`);
				},
				containment: "html",
				appendTo: ".canvas-main",	// 拖拽到
			});
		});

		// 鼠标在画板中移动事件
		canvasElem.on('mousemove', (e) => {
			const rect = e.currentTarget.getBoundingClientRect();
			this.mouseInCanvasPosition = {
				x: e.clientX - rect.left,
				y: e.clientY - rect.top
			};
		});

		// 鼠标离开画板事件
		canvasElem.on('mouseleave', (e) => {
			// 记录鼠标位置为离开状态
			this.mouseInCanvasPosition = { x: -1, y: -1 };
		});

		// 放置到画板中
		canvasElem.droppable({
			accept: ".tableview-item",	// 接收放置
			drop: (event, ui) => {		// 放置后回调
				if (ui.draggable.hasClass('tableview-item')) {
					// 计算窗体编号
					const paneId = Object.keys(this.paneDict).length > 0 ? Math.max(...Object.keys(this.paneDict)) + 1 : 1;
					// 创建窗体
					this.createPane(paneId, {
						type: 'table',
						name: $(ui.draggable).html(),
						subName: '',
						top: ui.position.top,
						left: ui.position.left,
						width: this.setting.paneWidth,
						height: this.setting.paneHeight
					}, (pane) => { pane.active(); });
				}
			}
		});

		// 底部配置栏缩放事件
		configurationElem.resizable({
			handles: 'n',
			minHeight: this.setting.resizable.configuration.min,
			maxHeight: this.setting.resizable.configuration.max,
			resize: (event, ui) => {
				this.designerElem.find('.canvas-view').css('bottom', (ui.size.height + 1) + 'px');
			}
		});

		// 底部配置栏切换事件
		configurationElem.find('.navbar-item').each((index, elem) => {
			const type = $(elem).attr('data-type');

			$(elem).on('click', () => {
				configurationElem.find('.navbar-item').removeClass('navbar-item-this');
				$(elem).addClass('navbar-item-this');
				configurationElem.find('.main-item').removeClass('main-item-this');
				configurationElem.find(`.main-item[data-type="${type}"]`).addClass('main-item-this');
			});
		});

		// 底部配置栏查询中添加点击事件
		configurationElem.find('.main-item[data-type="SELECT"] .tool .btn[data-type="create"]').on('click', () => {

		});

		// 底部配置栏查询中删除点击事件
		configurationElem.find('.main-item[data-type="SELECT"] .tool .btn[data-type="delete"]').on('click', () => {
			const content = this.designerElem.find('.configuration-view .main-item[data-type="SELECT"] .content');
			const elem = content.find('.content-active');
			const index = content.find('.content-item').index(elem);
			const paneId = elem.attr('data-pane');
			const fieldName = elem.attr('data-field');
			// 如果查询中有两个相同表或视图字段时，删除其中一个保留其窗体中字段勾选
			if (content.find(`.content-item[data-pane="${paneId}"][data-field="${fieldName}"]`).length == 1) {
				this.deleteConfigurationSelect(paneId, fieldName);
			}
			this.configuration.select.splice(index, 1);
			elem.remove();
		});

		// 底部配置栏查询中上移点击事件
		configurationElem.find('.main-item[data-type="SELECT"] .tool .btn[data-type="moveUp"]').on('click', () => {
			const content = this.designerElem.find('.configuration-view .main-item[data-type="SELECT"] .content');
			const elem = content.find('.content-active');
			const index = content.find('.content-item').index(elem);
			if (index == 0) return;
			const temp = this.configuration.select[index - 1];
			this.configuration.select[index - 1] = this.configuration.select[index];
			this.configuration.select[index] = temp;
			content.find('.content-item').eq(index - 1).before(elem);
		});

		// 底部配置栏查询中下移点击事件
		configurationElem.find('.main-item[data-type="SELECT"] .tool .btn[data-type="moveDown"]').on('click', () => {
			const content = this.designerElem.find('.configuration-view .main-item[data-type="SELECT"] .content');
			const elem = content.find('.content-active');
			const index = content.find('.content-item').index(elem);
			if (index == content.find('.content-item').length - 1) return;
			const temp = this.configuration.select[index + 1];
			this.configuration.select[index + 1] = this.configuration.select[index];
			this.configuration.select[index] = temp;
			content.find('.content-item').eq(index + 1).after(elem);
		});

		// 底部配置栏查询中去重复选框切换事件
		configurationElem.find('.main-item[data-type="SELECT"] .tool .btn[data-type="distinct"] input').on('change', () => {
			this.configuration.distinct = configurationElem.find('.main-item[data-type="SELECT"] #DISTINCT').prop('checked');
		});
	}

	/**
	 * 获取SQL语句
	 */
	getSQLSentence() {

	}

	/**
	 * 获取构建数据
	 */
	getConstructInfo() {
		return {
			pane: (() => {
				const temp = {};
				for (const paneId in this.paneDict) {
					const pane = this.paneDict[paneId];
					temp[paneId] = {
						type: pane.type,
						name: pane.name,
						subName: pane.subName,
						top: parseInt(pane.paneElem.css('top')),
						left: parseInt(pane.paneElem.css('left')),
						width: pane.paneElem.width(),
						height: pane.paneElem.height(),
						checkedFields: pane.checkedFields
					}
				}
				return temp;
			})(),
			line: (() => {
				const temp = {};
				for (const lineId in this.lineDict) {
					const line = this.lineDict[lineId];
					temp[lineId] = line.constructInfo;
				}
				return temp;
			})(),
			configuration: this.configuration
		};
	}

	/**
	 * 获取表和视图列表
	 */
	getTableListInfo(callback) {
		requestByIPC('dev/service/db/getTableViewList', {}, (req, res) => {
			if (res.code != 0) return this.showMessage(res.message);
			this.tableviewInfo = { tables: res.tables, views: res.views };
			callback && callback();
		});
	}

	/**
	 * 在画板中新增节点
	 * @param {Object} dom	新增节点的jq对象
	 */
	createDomFromCanvas(dom) {
		this.designerElem.find('.canvas-main').append(dom);
	}

	/**
	 * 画板中创建窗体
	 * @param {Object} 	 paneId				窗体编号
	 * @param {Object} 	 paneConstructInfo	窗体构建数据
	 * @param {Function} callback			创建完成后回调函数
	 */
	createPane(paneId, paneConstructInfo, callback) {
		const pane = new Pane(paneId, paneConstructInfo, this);
		this.paneDict[paneId] = pane;
		pane.init(() => { callback && callback(pane); });
	}

	/**
	 * 删除画板中窗体
	 * @param {Number} paneId	要删除窗体的编号
	 */
	removePane(paneId) {
		this.paneDict[paneId].paneElem.remove();
		delete this.paneDict[paneId];
	}

	/**
	 * 画板中创建连接线
	 * @param {Object} 	 lineId				连接线编号
	 * @param {Object} 	 lineConstructInfo	连接线构建数据
	 */
	createLine(lineId, lineConstructInfo) {
		const line = new Line(lineId, lineConstructInfo, this);
		this.lineDict[lineId] = line;
		line.init();
	}

	/**
	 * 删除画板中连接线
	 * @param {Number} lineId	要删除连接线的编号
	 */
	removeLine(lineId) {
		const line = this.lineDict[lineId];
		line.lineElem.remove();
		line.pointElemList.forEach(element => {
			element.remove();
		});
		delete line.paneList[0]?.lineDict[lineId];
		delete line.paneList[1]?.lineDict[lineId];
		delete this.lineDict[lineId];
	}

	/**
	 * 添加底部查询配置明细
	 * @param {Number} paneId		窗体编号
	 * @param {String} fieldName	字段名称
	 * @param {String} subFieldName	字段别名
	 * @param {String} isInsertData	是否添加到类属性中
	 */
	createConfigurationSelect(paneId, fieldName, subFieldName = '', isInsertData = true) {
		const pane = this.paneDict[paneId];
		if (isInsertData) this.configuration.select.push({ paneId: paneId, field: fieldName, subField: subFieldName });
		const elem = $(`
			<div class="content-item" data-pane="${paneId}" data-field="${fieldName}">
				<span class="table-name">${pane.subName || pane.name}.${fieldName}</span>
				<span class="sub-field-name">${subFieldName ? `(${subFieldName})` : ''}</span>
				<div class="delete-item">x</div>
			</div>
		`);
		this.designerElem.find('.configuration-view .main-item[data-type="SELECT"] .content').append(elem);

		elem.hover(() => {
			elem.find('.delete-item').show();
		}, () => {
			elem.find('.delete-item').hide();
		}).on('click', () => {
			this.designerElem.find('.configuration-view .main-item[data-type="SELECT"] .content .content-item').removeClass('content-active');
			elem.addClass('content-active');
		});

		elem.find('.delete-item').on('click', () => { this.deleteConfigurationSelect(paneId, fieldName); });
	}

	/**
	 * 删除底部查询配置明细
	 * @param {Number} paneId		窗体编号
	 * @param {String} fieldName	字段名称
	 */
	deleteConfigurationSelect(paneId, fieldName) {
		for (let i = 0; i < this.configuration.select.length; i++) {
			const item = this.configuration.select[i];
			if (item.paneId == paneId && item.field == fieldName) {
				this.configuration.select.splice(i, 1);
				this.designerElem.find('.configuration-view .main-item[data-type="SELECT"] .content .content-item').eq(i).remove();
				this.paneDict[paneId].changeFieldCheck(fieldName, false);
				break;
			}
		}
	}

	/**
	 * 提示信息
	 * @param {String} message	提示信息
	 */
	showMessage(message) {
		window.showMsg ? showMsg(message) : alert(message);
	}

	/**
	 * 弹出层
	 * @param {Any}		 elem				内容 jq对象,dom对象,html字符串
	 * @param {Object}	 config				配置项
	 * @param {String}	 config.title		标题
	 * @param {Any}	 	 config.top			上
	 * @param {Any}	 	 config.left		左
	 * @param {Number}	 config.width		宽
	 * @param {Number}	 config.height		高
	 * @param {String}	 config.teleport	挂载节点（选择器）
	 * @param {Function} confirm			点击确认后回调函数
	 */
	popup(elem, config, confirm) {
		if (typeof elem == 'string' || elem instanceof HTMLElement) elem = $(elem);
		if (typeof config.top == 'number') config.top += 'px';
		if (typeof config.left == 'number') config.left += 'px';
		const content = $(`
			<div class="view-design-popup" style="top:${config.top};left:${config.left};">
				${config.title ? `<div class="title">${config.title}</div>` : ''}
				<div class="content" style="width:${config.width}px;height:${config.height}px;"></div>
				<div class="toolbar">
					<div class="confirm">确认</div>
					<div class="cancel">取消</div>
				</div>
			</div>
		`);
		content.find('.content').append(elem);
		content.find('.toolbar .confirm').on('click', () => {
			confirm && confirm(content.find('.content'));
			content.remove();
		});
		content.find('.toolbar .cancel').on('click', () => {
			content.remove();
		});
		if (config.teleport) this.designerElem.find(config.teleport).append(content);
		else this.designerElem.append(content);
	}
}

/**
 * 可视化视图中窗体类
 */
class Pane {
	"use strict"

	/**
	 * 构造器
	 * @param {Number} id				窗体编号
	 * @param {Object} constructInfo	窗体构建数据
	 * @param {Object} viewDesign		编辑器对象
	 */
	constructor(id, constructInfo = {}, viewDesign) {
		// 窗体编辑设置
		this.setting = {
			top: 0,				// 默认上浮动
			left: 0,			// 默认左浮动
			width: 0,			// 默认窗体宽度
			height: 0,			// 默认窗体高度
			resizable: {		// 缩放
				width: {		// 左侧表和视图列表
					min: 200,	// 横向最小缩放
					max: 400	// 横向最大缩放
				},
				height: {		// 左侧表和视图列表
					min: 100,	// 纵向最小缩放
					max: 800	// 纵向最大缩放
				},
			}
		};
		// 窗体编号
		this.id = id;
		// 窗体构建数据
		this.constructInfo = constructInfo;
		// 编辑器对象
		this.viewDesign = viewDesign;
		// 当前窗体jq对象
		this.paneElem = null;
		// 常量字典
		this.DICT = {
			table: '表格',
			view: '视图',
			titleViewHeight: 36,
			fieldViewHeight: 24
		};
		// 表或视图	table,views
		this.type = '';
		// 表或视图名称
		this.name = '';
		// 表或视图别名
		this.subName = '';
		// 已选择字段
		this.checkedFields = [];
		// 字段数据
		this.fieldInfo = {};
		// 字段名列表
		this.fieldNameList = [];
		// 窗体关联线
		this.lineDict = {};
	}

	/**
	 * 初始化
	 * @param {Function} callback	完成后回调
	 */
	init(callback) {
		this.checkConstructInfo(() => {
			this.initPane(() => { this.bindEvent(); callback && callback(); });
		});
	}

	/**
	 * 校验窗体构建数据
	 * @param {Function} callback	完成后回调
	 */
	checkConstructInfo(callback) {
		if (!this.viewDesign) return console.error('未绑定视图编辑器！');
		this.setting.width = this.viewDesign.setting.paneWidth;
		this.setting.height = this.viewDesign.setting.paneHeight;
		if (!this.id) return console.error('未绑定窗体编号！');
		if (!/^\d+$/.test(this.id)) return console.error('绑定窗体编号无效！');
		if (!this.constructInfo?.name) return console.error('未绑定表或视图名称！');
		// 添加默认值
		this.constructInfo = {
			...{
				subName: '',
				top: 1,
				left: 1,
				width: this.setting.width,
				height: this.setting.height,
				checkedFields: []
			},
			...this.constructInfo
		}
		this.type = this.constructInfo.type;
		this.name = this.constructInfo.name;
		this.subName = this.constructInfo.subName;
		this.checkedFields = this.constructInfo.checkedFields;
		this.getFieldInfo(() => {
			// 判断勾选字段是否存在
			let temp = [];
			for (let i = 0; i < this.checkedFields.length; i++) {
				const item = this.checkedFields[i];
				if (this.fieldNameList.includes(item)) temp.push(item);
			}
			this.checkedFields = [...temp];
			callback && callback();
		});
	}

	/**
	 * 获取字段数据
	 * @param {Function} callback	完成后回调
	 */
	getFieldInfo(callback) {
		requestByIPC('dev/service/db/getTdmData', { tableName: this.constructInfo?.name }, (req, res) => {
			if (res.code != 0) return this.viewDesign.showMessage(res.message);
			const tdmData = JSON.parse(res.data);
			if (!tdmData?.fields) return this.viewDesign.showMessage(`未查询到${this.name}的字段！`);
			this.fieldInfo = { ...tdmData.fields };
			this.fieldNameList = Object.keys(tdmData.fields);
			callback && callback();
		});
	}

	/**
	 * 绘制窗体
	 * @param {Function} callback	完成后回调
	 */
	initPane(callback) {
		const paneElem = $(`
			<div class="pane-box ui-widget-content" data-type="" data-id="">
				<div class="header">
					<span class="title"></span>
					<span class="subtitle" style="display:none;"></span>
				</div>
				<div class="fields"></div>
			</div>
		`);

		// 设置窗体显示属性
		paneElem.attr({
			'data-type': this.constructInfo.type,
			'data-id': this.id,
		}).css({
			'top': this.constructInfo.top + 'px',
			'left': this.constructInfo.left + 'px',
			'width': this.constructInfo.width + 'px',
			'height': this.constructInfo.height + 'px',
		});

		paneElem.find('.header .title').html(this.constructInfo.name);
		this.constructInfo.subName && paneElem.find('.header .subtitle').show().html(`(${this.constructInfo.subName})`);

		this.viewDesign.createDomFromCanvas(paneElem);
		this.paneElem = $(`.pane-box[data-id="${this.id}"]`);

		// 初始化窗体中字段
		this.initFields(callback);
	}

	/**
	 * 初始化窗体中字段
	 * @param {Function} callback	完成后回调
	 */
	initFields(callback) {
		const tempFieldElem = $(`
			<div class="field" data-name="">
				<input type="checkbox">
				<label></label>
			</div>
		`);

		// 添加全选字段到窗体中
		const fieldAllElem = tempFieldElem.clone();
		fieldAllElem.attr('data-name', '*');
		fieldAllElem.find('label').html('*');
		fieldAllElem.find('span').html('*');
		if (this.constructInfo.checkedFields.includes('*')) fieldAllElem.find('input').prop('checked', true);
		this.paneElem.find('.fields').append(fieldAllElem);

		// 添加字段到窗体中
		for (const fieldName in this.fieldInfo) {
			const fieldElem = tempFieldElem.clone();
			fieldElem.attr('data-name', fieldName);
			fieldElem.find('label').html(fieldName);
			fieldElem.find('span').html(fieldName);
			if (this.constructInfo.checkedFields.includes(fieldName)) fieldElem.find('input').prop('checked', true);
			this.paneElem.find('.fields').append(fieldElem);
		}

		callback && callback();
	}

	/**
	 * 绑定窗体事件
	 */
	bindEvent() {
		// 窗体鼠标按下事件
		this.paneElem.on('mousedown', () => { this.active(); });

		// 窗体右键事件
		this.paneElem.find('.header').TBContextmenu([
			{
				"type": 'updateSubName',
				"title": '编辑别名',
				"data": null,
				"event": () => {
					this.updateSubName();
				}
			},
			{
				"type": 'remove',
				"title": '移除' + this.DICT[this.constructInfo.type],
				"data": null,
				"event": () => {
					this.remove();
				}
			},
			{
				"type": 'reload',
				"title": '刷新' + this.DICT[this.constructInfo.type],
				"data": null,
				"event": () => {
					this.reload();
				}
			}
		], null);

		// 窗体缩放事件
		this.paneElem.resizable({
			handles: 'all',									// 可缩放方式
			minWidth: this.setting.resizable.width.min,		// 最小缩小宽度
			maxWidth: this.setting.resizable.width.max,		// 最大放大宽度
			minHeight: this.setting.resizable.height.min,	// 最小缩小高度
			maxHeight: this.setting.resizable.height.max,	// 最大放大高度
			resize: () => {						// 移动时
				for (const lineId in this.lineDict) {
					this.lineDict[lineId].updatePointAndLinePosition();
				}
			}
		});

		// 窗体移动事件
		this.paneElem.draggable({
			containment: ".canvas-main",	// 移动范围
			handle: '.header',				// 可拖拽节点
			scroll: true,					// 允许滚动条滚动
			drag: () => {					// 移动时
				for (const lineId in this.lineDict) {
					this.lineDict[lineId].updatePointAndLinePosition();
				}
			}
		});

		// 字段列表滚动事件
		this.paneElem.find('.fields').scroll(() => {
			for (const lineId in this.lineDict) {
				this.lineDict[lineId].updatePointAndLinePosition();
			}
		});

		// 绑定字段事件
		this.bindFieldsEvent();
	}

	/**
	 * 绑定字段事件
	 */
	bindFieldsEvent() {
		this.paneElem.find('.field').each((index, elem) => {
			const self = this;
			elem = $(elem);
			const fieldName = elem.attr('data-name');

			// 字段选择事件
			elem.on('click', () => {
				elem.find('input').prop('checked', !elem.find('input').prop('checked'));
				if (elem.find('input').prop('checked') && !this.checkedFields.includes(fieldName)) {
					this.checkedFields.push(fieldName);
					this.viewDesign.createConfigurationSelect(this.id, fieldName);
				}
				else {
					this.checkedFields.splice(this.checkedFields.indexOf(fieldName), 1);
					this.viewDesign.deleteConfigurationSelect(this.id, fieldName);
				}
			});

			// 字段全选
			if (fieldName == '*') return true;

			// 字段拖拽事件
			elem.draggable({
				cursor: "move",				// 拖拽方式
				revert: "invalid",			// 还原方式
				helper: '',					// 视觉反馈
				appendTo: ".canvas-main",	// 拖拽到
				start: () => {				// 拖拽开始
					const fieldName = elem.attr('data-name');
					const mouseInCanvasPosition = this.viewDesign.mouseInCanvasPosition;
					const line = new Line(0, [
						{ paneId: this.id, field: fieldName },
						mouseInCanvasPosition,
					], this.viewDesign);
					this.viewDesign.lineDict[0] = line;
					line.init();
				},
				drag: () => {				// 拖拽中
					const mouseInCanvasPosition = this.viewDesign.mouseInCanvasPosition;
					if (mouseInCanvasPosition.x != -1 || mouseInCanvasPosition.y != -1) this.viewDesign.lineDict[0] && this.viewDesign.lineDict[0].updateTempPointAndTempLinePosition();
				},
				stop: () => {				// 拖拽结束
					this.viewDesign.lineDict[0] && this.viewDesign.lineDict[0].remove();
				}
			});

			// 放置到字段中
			elem.droppable({
				accept: ".field",		// 接收放置
				drop: (event, ui) => {	// 放置后回调
					createLineOnDrop(ui);
				}
			});

			// 放置到字段中
			elem.find('label').droppable({
				accept: ".field",		// 接收放置
				drop: (event, ui) => {	// 放置后回调
					createLineOnDrop(ui);
				}
			});

			function createLineOnDrop(ui) {
				// 放置时计算来源字段节点和当前字段节点字段并连线
				const _paneId = $(ui.draggable).parents('.pane-box').attr('data-id');
				const _fieldName = $(ui.draggable).attr('data-name');
				const lineId = Object.keys(self.viewDesign.lineDict).length > 0 ? Math.max(...Object.keys(self.viewDesign.lineDict)) + 1 : 1;
				self.viewDesign.createLine(lineId, [
					{ paneId: self.id, field: fieldName },
					{ paneId: _paneId, field: _fieldName },
				]);
			}
		});
	}

	/**
	 * 选中窗体
	 */
	active() {
		this.viewDesign.designerElem.find('.pane-box').removeClass('pane-active');
		this.paneElem.addClass('pane-active');
	}

	/**
	 * 刷新窗体
	 */
	reload() {
		this.paneElem.find('.fields').empty();
		// 初始化窗体中字段
		this.initFields(() => {
			// 绑定字段事件
			this.bindFieldsEvent();
		});
	}

	/**
	 * 移除窗体
	 */
	remove() {
		this.viewDesign.removePane(this.id);
	}

	/**
	 * 编辑别名
	 */
	updateSubName() {
		this.viewDesign.popup($('<input>').css({
			border: '0px',
			color: 'black',
			width: 'calc(100% - 10px)',
			height: '24px',
			margin: '5px',
			'border-radius': '2px'
		}).val(this.subName || ''), {
			title: '编辑别名',
			top: this.getTop() + 36,
			left: this.getLeft() + this.getWidth() / 2,
			width: 200,
			height: 36,
			teleport: '.canvas-view'
		}, (elem) => {
			// 修改当前窗体别名
			this.subName = elem.find('input').val() || '';
			this.paneElem.find('.subtitle').html(this.subName ? '&nbsp;(' + this.subName + ')' : '').show();
			// 修改底部查询配置与当前窗体相关的别名
			for (let i = 0; i < this.viewDesign.configuration.select.length; i++) {
				const item = this.viewDesign.configuration.select[i];
				if (item.paneId != this.id) continue;
				this.viewDesign.designerElem.find('.main-item[data-type="SELECT"] .content .content-item').eq(i).find('.table-name').html(this.subName ? this.subName + '.' + item.field : this.name);
			}
		});
	}

	/**
	 * 获取字段在窗体的位置
	 * 1. 根据窗体字段列表滚动高度和窗体高度计算出当前字段是否在可视区域
	 * 2. 如果字段在可视区域返回字段在画板中横纵坐标
	 * 3. 否则返回与可视区域相邻的上下一条边在画板中横纵坐标
	 * @param  {String} fieldName	字段名称
	 * @return {Object}
	 */
	getFieldPositionInPane(fieldName) {
		// 字段序号
		const fieldIndex = this.fieldNameList.indexOf(fieldName);
		// 校验字段是否存在
		if (fieldIndex == -1) return false;
		// 窗体横坐标
		const panePositionx = parseInt(this.paneElem.css('left'));
		// 窗体纵坐标
		const panePositiony = parseInt(this.paneElem.css('top'));
		// 窗体宽度
		const paneWidth = parseInt(this.paneElem.width());
		// 窗体高度
		const paneHeight = parseInt(this.paneElem.height());
		// 字段列表高度
		const fieldsHeight = this.paneElem.find('.fields').height();
		// 字段列表滚动高度
		const fieldsScrollY = this.paneElem.find('.fields').scrollTop();
		// 字段所在高度
		const fieldOffsetY = (fieldIndex + 1) * this.DICT.fieldViewHeight;

		// 字段在可视区域上方
		if (fieldsScrollY > fieldOffsetY) {
			return {
				lx: panePositionx,
				rx: panePositionx + paneWidth,
				y: panePositiony + this.DICT.titleViewHeight + 1
			}
		}

		// 字段在可视区域中
		if (fieldsScrollY <= fieldOffsetY && fieldOffsetY <= fieldsScrollY + fieldsHeight) {
			return {
				lx: panePositionx,
				rx: panePositionx + paneWidth,
				y: panePositiony + this.DICT.titleViewHeight + fieldOffsetY - fieldsScrollY + 8
			}
		}

		// 字段在可视区域下方
		if (fieldOffsetY > fieldsScrollY + fieldsHeight) {
			return {
				lx: panePositionx,
				rx: panePositionx + paneWidth,
				y: panePositiony + paneHeight - 4
			}
		}
	}

	/**
	 * 获取与其他窗体位置关系
	 * @param  {Number} paneId	窗体编号
	 * @return {String}
	 */
	getOrientationToOtherPane(paneId) {
		// 窗体横坐标
		const panePositionx = parseInt(this.paneElem.css('left'));
		// 窗体宽度
		const paneWidth = parseInt(this.paneElem.width());
		// 窗体横坐标
		const _panePositionx = parseInt(this.viewDesign.paneDict[paneId].paneElem.css('left'));
		// 窗体宽度
		const _paneWidth = parseInt(this.viewDesign.paneDict[paneId].paneElem.width());
		return panePositionx + paneWidth / 2 >= _panePositionx + _paneWidth / 2 ? 'left' : 'right';
	}

	/**
	 * 获取与鼠标位置关系
	 * @retrun 			left,right
	 */
	getOrientationToMouse() {
		// 当前鼠标在窗体的位置
		const mouseInCanvasPosition = this.viewDesign.mouseInCanvasPosition;
		// 窗体横坐标
		const panePositionx = parseInt(this.paneElem.css('left'));
		// 窗体宽度
		const paneWidth = parseInt(this.paneElem.width());

		return panePositionx + paneWidth / 2 >= mouseInCanvasPosition.x ? 'left' : 'right';
	}

	/**
	 * 修改字段选择
	 * @param {String}  fieldName	字段名称	
	 * @param {Boolean} isCheck		是否选择
	 */
	changeFieldCheck(fieldName, isCheck) {
		const index = this.fieldNameList.indexOf(fieldName);
		if (index == -1) return console.error('未找到字段！');
		this.paneElem.find('.fields .field').eq(index + 1).find('input').prop('checked', isCheck);
		this.checkedFields.splice(this.checkedFields.indexOf(fieldName), 1);
	}

	/**
	 * 获取窗体宽度
	 */
	getWidth() { return parseInt(this.paneElem.width()); }

	/**
	 * 获取窗体高度
	 */
	getHeight() { return parseInt(this.paneElem.height()); }

	/**
	 * 获取窗体上方浮动
	 */
	getTop() { return parseInt(this.paneElem.css('top')); }

	/**
	 * 获取窗体左侧浮动
	 */
	getLeft() { return parseInt(this.paneElem.css('left')); }
}

/**
 * 可视化视图中连接线类
 */
class Line {
	"use strict"

	/**
	 * 构造器
	 * @param {Number} id				连接线编号
	 * @param {Object} constructInfo	连接线构建数据
	 * @param {Object} viewDesign		编辑器对象
	 */
	constructor(id = 0, constructInfo = {}, viewDesign) {
		// 连接线编辑设置
		this.setting = {};
		// 连接线编号
		this.id = id;
		// 连接线构建数据
		this.constructInfo = constructInfo;
		// 编辑器对象
		this.viewDesign = viewDesign;
		// 窗体对象列表
		this.paneList = [null, null];
		// 锚点jq对象列表
		this.pointElemList = [null, null];
		// 连接线jq对象
		this.lineElem = null;
	}

	/**
	 * 初始化
	 * @param {Function} callback	完成后回调
	 */
	init(callback) {
		this.checkConstructInfo(() => {
			if (this.id == 0) this.initTempLine();
			else this.initLine();
			callback && callback();
		});
	}

	/**
	 * 校验连接线构建数据
	 * @param {Function} callback	完成后回调
	 */
	checkConstructInfo(callback) {
		this.constructInfo.length = 2;
		if (!this.viewDesign) return console.error('未绑定视图编辑器！');
		if (this.id == undefined || this.id == null) return console.error('未绑定连接线编号！');
		if (!/^\d+$/.test(this.id)) return console.error('绑定连接线编号无效！');
		if (!this.constructInfo[0].paneId || !/^\d+$/.test(this.constructInfo[0].paneId)) return console.error('绑定连接窗体编号无效！');
		if (!this.constructInfo[0].field || !this.viewDesign.paneDict[this.constructInfo[0].paneId].fieldNameList.includes(this.constructInfo[0].field)) return console.error('绑定连接窗体字段无效！');
		if (this.id > 0 && (!this.constructInfo[1].paneId || !/^\d+$/.test(this.constructInfo[1].paneId))) return console.error('绑定连接窗体编号无效！');
		if (this.id > 0 && (!this.constructInfo[1].field || !this.viewDesign.paneDict[this.constructInfo[0].paneId].fieldNameList.includes(this.constructInfo[0].field))) return console.error('绑定连接窗体字段无效！');
		callback && callback();
	}

	/**
	 * 绘制连接线
	 * 1. 根据连接线构建数据找到需要连接的两个窗体中的字段
	 * 2. 计算两个字段的位置信息
	 * 3. 分别判断字段位置是否展示，如果不是计算窗体底部位置
	 * 4. 根据计算出的两点位置连线
	 * @param {Function} callback	完成后回调
	 */
	initLine(callback) {
		for (const i of [0, 1]) {
			this.paneList[i] = this.viewDesign.paneDict[this.constructInfo[i].paneId];
			this.paneList[i].lineDict[this.id] = this;
			const orientation = this.paneList[i].getOrientationToOtherPane(this.constructInfo[Math.abs(i - 1)].paneId);
			const position = this.paneList[i].getFieldPositionInPane(this.constructInfo[i].field);
			position.x = orientation == 'left' ? position.lx : position.rx;
			this.pointElemList[i] = this.drawPoint(orientation, position);
		}
		this.drawLine({
			x: parseInt(this.pointElemList[0].css('left')),
			y: parseInt(this.pointElemList[0].css('top'))
		}, {
			x: parseInt(this.pointElemList[1].css('left')),
			y: parseInt(this.pointElemList[1].css('top'))
		});

		callback && callback();
	}

	/**
	 * 绘制临时连接线
	 */
	initTempLine() {
		this.paneList[0] = this.viewDesign.paneDict[this.constructInfo[0].paneId];
		const orientation = this.paneList[0].getOrientationToMouse();
		const position = this.paneList[0].getFieldPositionInPane(this.constructInfo[0].field);
		position.x = orientation == 'left' ? position.lx : position.rx;
		this.pointElemList[0] = this.drawPoint(orientation, position);
		this.pointElemList[1] = this.drawPoint(orientation == 'left' ? 'right' : 'left', this.viewDesign.mouseInCanvasPosition);
		this.drawLine({
			x: parseInt(this.pointElemList[0].css('left')),
			y: parseInt(this.pointElemList[0].css('top'))
		}, {
			x: parseInt(this.pointElemList[1].css('left')),
			y: parseInt(this.pointElemList[1].css('top'))
		});
	}

	/**
	 * 绑定连接线事件
	 */
	bindEvent() {
		// 连接线鼠标按下事件
		this.lineElem.on('mousedown', () => { this.active(); });

		// 窗体右键事件
		this.lineElem.TBContextmenu([
			{
				"type": 'remove',
				"title": '移除',
				"data": null,
				"event": () => {
					this.remove();
				}
			}
		], null);
	}

	/**
	 * 添加锚点
	 * @param {String}	orientation	方向 [left,right]
	 * @param {Object}	position	位置
	 * @param {Number}	position.x	纵坐标
	 * @param {Number}	position.y	横坐标
	 * @retrun 
	 */
	drawPoint(orientation, position) {
		if (orientation != 'left' && orientation != 'right') orientation = 'right';
		let view = null;
		if (orientation == 'left') view = $(`<div class="arrow-left" style="top:${position.y}px;left:${position.x - 4}px;"></div>`);
		if (orientation == 'right') view = $(`<div class="arrow-right" style="top:${position.y}px;left:${position.x}px;"></div>`);
		view.attr('data-type', orientation);
		this.viewDesign.createDomFromCanvas(view);
		return view;
	}

	/**
	 * 两点间连线
	 * @param {Object}	position1		起点对象
	 * @param {Number}	position1.x		起点横坐标
	 * @param {Number}	position1.y		起点纵坐标
	 * @param {Object}	position2		终点对象
	 * @param {Number}	position2.x		终点横坐标
	 * @param {Number}	position2.y		终点纵坐标
	 */
	drawLine(position1, position2) {
		if (position1.x < position2.x) position1.x += 5;
		if (position1.x > position2.x) position1.x += 1;
		if (position2.x < position1.x) position2.x += 5;
		if (position2.x > position1.x) position2.x += 1;
		position1.y += 4;
		position2.y += 4;

		// 用勾股定律计算出斜边长度及其夹角（即连线的旋转角度）
		const lx = position2.x - position1.x;
		const ly = position2.y - position1.y;
		// 计算连线长度
		const length = Math.sqrt(lx * lx + ly * ly);
		// 弧度值转换为角度值
		const c = 360 * Math.atan2(ly, lx) / (2 * Math.PI);
		const midX = (position2.x + position1.x) / 2;
		const midY = (position2.y + position1.y) / 2;
		const deg = c <= -90 ? (360 + c) : c;  // 负角转换为正角

		const line = $('<div>', { class: 'line' }).css({
			top: midY + 'px',
			left: (midX - length / 2) + 'px',
			width: length + 'px',
			transform: 'rotate(' + deg + 'deg)',
		});
		this.viewDesign.createDomFromCanvas(line);
		this.lineElem = line;
		this.bindEvent();
		return line;
	}

	/**
	 * 修改锚点与连接线的位置
	 */
	updatePointAndLinePosition() {
		for (const i of [0, 1]) {
			this.pointElemList[i].remove();
			this.pointElemList[i] = null;
			const orientation = this.paneList[i].getOrientationToOtherPane(this.constructInfo[Math.abs(i - 1)].paneId);
			const position = this.paneList[i].getFieldPositionInPane(this.constructInfo[i].field);
			position.x = orientation == 'left' ? position.lx : position.rx;
			this.pointElemList[i] = this.drawPoint(orientation, position);
		}
		this.lineElem.remove();
		this.lineElem = null;
		this.lineElem = this.drawLine({
			x: parseInt(this.pointElemList[0].css('left')),
			y: parseInt(this.pointElemList[0].css('top'))
		}, {
			x: parseInt(this.pointElemList[1].css('left')),
			y: parseInt(this.pointElemList[1].css('top'))
		});
	}

	/**
	 * 修改临时锚点与临时连接线的位置
	 */
	updateTempPointAndTempLinePosition() {
		this.pointElemList[0].remove();
		this.paneList[0] = this.viewDesign.paneDict[this.constructInfo[0].paneId];
		const orientation = this.paneList[0].getOrientationToMouse();
		const position = this.paneList[0].getFieldPositionInPane(this.constructInfo[0].field);
		position.x = orientation == 'left' ? position.lx : position.rx;
		this.pointElemList[0] = this.drawPoint(orientation, position);
		this.pointElemList[1].remove();
		this.pointElemList[1] = this.drawPoint(orientation == 'left' ? 'right' : 'left', this.viewDesign.mouseInCanvasPosition);

		this.lineElem.remove();
		this.lineElem = null;
		this.lineElem = this.drawLine({
			x: parseInt(this.pointElemList[0].css('left')),
			y: parseInt(this.pointElemList[0].css('top'))
		}, {
			x: parseInt(this.pointElemList[1].css('left')),
			y: parseInt(this.pointElemList[1].css('top'))
		});
	}

	/**
	 * 选中窗体
	 */
	active() {
		this.viewDesign.designerElem.find('.line').removeClass('line-active');
		this.lineElem.addClass('line-active');
	}

	/**
	 * 移除连接线
	 */
	remove() {
		this.viewDesign.removeLine(this.id);
	}
}